# 機能設計書 48-Pydantic v2互換

## 概要

本ドキュメントは、FastAPIにおけるPydantic v2互換機能の設計と実装について記述する。この機能はPydantic v2との互換性レイヤーを提供し、モデルフィールド、バリデーション、シリアライゼーションの統合を実現する。

### 本機能の処理概要

**業務上の目的・背景**：Pydanticはデータバリデーションとシリアライゼーションのためのライブラリであり、FastAPIの中核的な依存関係である。Pydantic v1からv2への移行に伴い、内部APIに大幅な変更が加わった。FastAPIはこれらの変更に対応しつつ、ユーザーに対して一貫したインターフェースを提供する必要がある。_compatモジュールはこの互換性レイヤーとして機能する。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- リクエストボディのバリデーション
- レスポンスモデルのシリアライゼーション
- OpenAPIスキーマの生成
- フォームデータの解析
- 型アノテーションの解析と検証

**主要な処理内容**：
1. Pydanticバージョンの検出と分岐処理
2. ModelFieldクラスによるフィールド抽象化
3. 型アノテーションの解析（Union、シーケンス、UploadFile等）
4. JSON Schema生成の統合
5. バリデーションとシリアライゼーションの実行

**関連システム・外部連携**：Pydantic v2、pydantic-core、typing_extensions

**権限による制御**：本機能自体は権限制御を行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は内部互換性レイヤーであり、直接関連する画面はない |

## 機能種別

互換性レイヤー / バリデーション / シリアライゼーション / 型処理

## 入力仕様

### 主要なエクスポート関数・クラス

| 名前 | 型 | 説明 |
|-----|-----|------|
| ModelField | class | フィールド情報を抽象化するデータクラス |
| BaseConfig | class | Pydantic設定の基底クラス（現在は空実装） |
| Undefined | PydanticUndefined | 未定義値を表す特殊値 |
| UndefinedType | PydanticUndefinedType | Undefinedの型 |
| Validator | Any | バリデータの型エイリアス |
| RequiredParam | PydanticUndefined | 必須パラメータを表す値 |

### shared.pyの主要関数

| 関数名 | 説明 |
|--------|------|
| lenient_issubclass | 型エラーを無視するissubclass |
| field_annotation_is_scalar | アノテーションがスカラー型か判定 |
| field_annotation_is_sequence | アノテーションがシーケンス型か判定 |
| value_is_sequence | 値がシーケンスか判定 |
| is_uploadfile_or_nonable_uploadfile_annotation | UploadFile型か判定 |
| annotation_is_pydantic_v1 | Pydantic v1モデルか判定 |

### 入力データソース

FastAPIの依存性注入システム、パラメータ解析システムからの呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| validated_data | Any | バリデーション済みデータ |
| serialized_data | Any | シリアライズされたデータ |
| json_schema | dict | JSON Schema |
| errors | list[dict] | バリデーションエラー情報 |

### 出力先

FastAPIの各処理層（ルーティング、依存性解決、OpenAPI生成等）

## 処理フロー

### 処理シーケンス（ModelField使用例）

```
1. ModelField作成
   └─ FieldInfo、name、modeを指定
2. __post_init__実行
   └─ TypeAdapterの初期化
3. validate呼び出し
   └─ 入力値のバリデーション
4. serialize呼び出し
   └─ 出力値のシリアライゼーション
```

### フローチャート

```mermaid
flowchart TD
    A[FastAPIパラメータ解析] --> B[ModelField作成]
    B --> C[TypeAdapter初期化]
    C --> D{バリデーション実行}
    D -->|成功| E[バリデーション済み値返却]
    D -->|失敗| F[エラー情報返却]
    E --> G[シリアライゼーション]
    G --> H[レスポンス生成]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-48-01 | v1非サポート | Pydantic v1モデルはFastAPIでサポートされない | v1モデル検出時 |
| BR-48-02 | バリデーションモード | validationまたはserializationモードを選択可能 | ModelField作成時 |
| BR-48-03 | エラーロケーション | バリデーションエラーにはロケーション情報が付与される | エラー発生時 |
| BR-48-04 | 警告抑制 | Pydantic 2.12以降のUnsupportedFieldAttributeWarningは抑制 | 該当バージョン時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | PydanticV1NotSupportedError | Pydantic v1モデルが使用された | v2モデルに更新 |
| - | PydanticSchemaGenerationError | スキーマ生成に失敗 | 型定義を確認 |
| - | ValidationError | バリデーション失敗 | 入力データを確認 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- TypeAdapterは初期化時にスキーマをコンパイルし、以降はキャッシュを使用
- get_cached_model_fieldsはlru_cacheでキャッシュ
- WeakKeyDictionaryで型キャッシュを管理しメモリリークを防止

## セキュリティ考慮事項

- バリデーションにより不正な入力データを排除
- シリアライゼーション時にセンシティブ情報のフィルタリングが可能

## 備考

- _compatディレクトリは`__init__.py`、`shared.py`、`v2.py`の3ファイルで構成
- shared.pyは型判定のユーティリティ関数を提供
- v2.pyはPydantic v2固有の実装を提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

パッケージ構成と公開インターフェースを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | __init__.py | `fastapi/_compat/__init__.py` | 公開される関数・クラスの一覧 |

**読解のコツ**: __init__.pyはshared.pyとv2.pyから必要な要素を再エクスポートしている。42行にわたるインポート文で全体像を把握できる。

#### Step 2: 共通ユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | shared.py | `fastapi/_compat/shared.py` | 型判定ユーティリティ関数 |

**主要処理フロー**:
- **30-31行目**: PYDANTICバージョン検出
- **49-57行目**: lenient_issubclass関数
- **60-75行目**: シーケンス型判定関数群
- **139-147行目**: UploadFile型判定
- **194-206行目**: Pydantic v1アノテーション判定

#### Step 3: Pydantic v2統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | v2.py | `fastapi/_compat/v2.py` | ModelFieldクラスとスキーマ生成 |

**主要処理フロー**:
- **44-48行目**: Pydanticから再エクスポートされる値（RequiredParam、Undefined等）
- **98-212行目**: ModelFieldデータクラス
- **221-313行目**: get_definitions関数（JSON Schema生成）
- **452-458行目**: create_body_model関数

### プログラム呼び出し階層図

```
FastAPI パラメータ解析
    │
    └─ _compat モジュール
           │
           ├─ shared.py
           │      ├─ lenient_issubclass
           │      ├─ field_annotation_is_scalar
           │      ├─ field_annotation_is_sequence
           │      └─ annotation_is_pydantic_v1
           │
           └─ v2.py
                  ├─ ModelField
                  │      ├─ validate()
                  │      └─ serialize()
                  │
                  ├─ get_definitions()
                  ├─ get_schema_from_model_field()
                  └─ create_body_model()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

リクエストデータ ────────▶ ModelField.validate() ────────▶ バリデーション済みデータ
       │                      │                              │
       └─ JSON/Form          └─ TypeAdapter.validate_python  └─ validated value or errors

モデルデータ ────────────▶ ModelField.serialize() ────────▶ シリアライズ済みデータ
       │                      │                              │
       └─ Python object      └─ TypeAdapter.dump_python      └─ JSON-compatible value
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| __init__.py | `fastapi/_compat/__init__.py` | ソース | 公開インターフェース定義 |
| shared.py | `fastapi/_compat/shared.py` | ソース | 共通型判定ユーティリティ |
| v2.py | `fastapi/_compat/v2.py` | ソース | Pydantic v2統合実装 |
| utils.py | `fastapi/utils.py` | ソース | _compatを使用するユーティリティ |
| params.py | `fastapi/params.py` | ソース | パラメータ定義での使用 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | 依存性解決での使用 |
